/*
 * 07/08/2001 - 19:44:29
 *
 * Player.java - A pure Java OGG player
 * Copyright (C) 2001 Romain Guy
 * Some parts are (C) JCraft team and Tom Gutwin
 * romain.guy@jext.org
 * www.jext.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package org.jext.oggre.player;

import java.io.*;
import java.net.*;

import com.jcraft.jorbis.*;
import com.jcraft.jogg.*;

import javax.sound.midi.*;
import javax.sound.sampled.*;

import org.jext.oggre.playlist.PlayList;
import org.jext.oggre.playlist.PlayListFactory;
import org.jext.oggre.playlist.Song;

/**
 * <code>Player</code> is the main class of the <b>OGGre</b> OGG player.<p>
 * This class provides methods to play files (play, stop, pause...).<p>
 * Parts of work based on JCraft crew and Tom Gutwin code.
 * @author Romain Guy
 */

public class Player implements Runnable
{
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PRIVATE FIELDS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  // the playlist
  private PlayList playList;
  // the song
  private Song song;
  // volume
  private int volume = 5;
  // volume multiplier
  private double volumeMultiplier;
  // pause
  private boolean paused = false;
  // timer listener
  private OGGListener listener = null;
  // timer
  private Timer timer;

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // CONSTRUCTORS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Creates a new player. This player receive an empty associated playlist.
   */

  public Player()
  {
    this(new PlayList());
  }

  /**
   * Creates a new player, associated to a given playlist.
   * @param playList The playlist used to play songs
   */

  public Player(PlayList playList)
  {
    this.playList = playList;
    setVolume(5);
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PLAY RELATED METHODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Plays the next song in the playlist.
   * @return A boolean value which defines the succes of the operation
   */

  public boolean next()
  {
    song = playList.next();
    if (song != null)
    {
      startPlay();
      return true;
    } else {
      stop();
      return false;
    }
  }

  /**
   * Pauses the play of the current song.
   * @return A boolean value which defines the succes of the operation
   */

  public boolean pausePlay()
  {
    if (!paused)
    {
      outputLine.stop();
      paused = true;
      return true;
    } else
      return false;
  }

  /**
   * Starts playing current song in the associated playlist.
   * @return A boolean which values defines the success of the operation
   */

  public boolean play()
  {
    if (paused)
    {
      paused = false;
      outputLine.start();
      return true;
    }

    if (player == null && playList.isShuffleEnabled())
      song = playList.next();
    else
      song = playList.getCurrentSong();
    
    if (song != null)
    {
      startPlay();
      return true;
    } else
      return false;
  }

  /**
   * Plays the previous song in the playlist.
   * @return A boolean value which defines the succes of the operation
   */

  public boolean previous()
  {
    song = playList.previous();
    if (song != null)
    {
      startPlay();
      return true;
    } else {
      stop();
      return false;
    }
  }

  /**
   * Stops the play of the current song.
   * @return A boolean value which defines the succes of the operation
   */

  public boolean stopPlay()
  {
    player = null;
    stop();
    return true;
  }

  // do the start playing job

  private void startPlay()
  {
    if (player != null)
      stopPlay();

    bitStream = null;
    try
    {
      URL url = song.getLocation();
      URLConnection urlc = url.openConnection();
      bitStream = urlc.getInputStream();
    } catch (Exception ee) { }

    System.out.println("Playing: " + song.getName());

    if (bitStream == null)
      System.out.println("Cannot find song !");
    else
    {
      paused = false;

      if (listener != null)
        timer = new Timer();

      player = new Thread(this);
      player.start();
    }
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // VOLUME RELATED METHODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the sound volume.
   */

  public int getVolume()
  {
    return volume;
  }

  /**
   * Sets the sound volume.
   * @param volume The new volume
   */

  public void setVolume(int volume)
  {
    if (volume >= 0 && volume <= 10)
    {
      this.volumeMultiplier = volumeMultiplier(volume);
      this.volume = volume;
    }
  }

  /**
   * Converts a linear mapped value to its Log equivalent. For use
   * in the Volume Scale.
   * @param currSetting is the linear value to convert.
   *                    Acceptable values: 0 = mute, 10 = full
   * @return the Log value (Log currSetting)
   *
   */

  private double volumeMultiplier(int currSetting)
  {
    double retVal = 0.0;
    double [] logLookup = { 0.0000, 0.0400, 0.3010, 0.4771, 0.6021, 0.6990,
                            0.7781, 0.8451, 0.9031, 0.9542, 1.0000, 1.1000 };
    double [] inverseLogLookup = { 0.0000, 1 - logLookup[9], 1 - logLookup[8],
                                   1 - logLookup[7], 1 - logLookup[6], 1 - logLookup[5],
                                   1 - logLookup[4],
                                   1 - logLookup[3], 1 - logLookup[2], 1 - logLookup[1],
                                   1.0000, 1.1000};

    if (currSetting > 10)
      retVal = 1.1;
    else if (currSetting > 0)
    {
      //if (logVolumeScale_)
      retVal = inverseLogLookup[currSetting];
      //else
      //  retVal = (double) currSetting / 10;
    }

    return retVal;
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PLAYLIST RELATED METHODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the playlist associated to this player.
   */

  public PlayList getPlayList()
  {
    return playList;
  }

  /**
   * Changes the associated playlist.
   * @param playList The new playlist
   */

  public void setPlayList(PlayList playList)
  {
    this.playList = playList;
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // OGG PLAYER
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  Thread player = null;
  InputStream bitStream = null;

  static final int BUFSIZE = 4096 * 2;
  static int convsize = BUFSIZE * 2;
  static byte[] convbuffer = new byte[convsize];

  SyncState oy;
  StreamState os;
  Page og;
  Packet op;
  Info vi;
  Comment vc;
  DspState vd;
  Block vb;

  byte[] buffer = null;
  int bytes = 0;

  int format;
  int rate = 0;
  int channels = 0;
  int left_vol_scale = 100;
  int right_vol_scale = 100;
  SourceDataLine outputLine = null;

  int frameSizeInBytes;
  int bufferLengthInBytes;

  int lastSecond;
  int second;

  // init jorbis lib

  private void init_jorbis()
  {
    oy = new SyncState();
    os = new StreamState();
    og = new Page();
    op = new Packet();

    vi = new Info();
    vc = new Comment();
    vd = new DspState();
    vb = new Block(vd);

    buffer = null;
    bytes = 0;
    lastSecond = 0;
    second = 0;

    oy.init();
  }

  // reads into the music stream

  private SourceDataLine getOutputLine(int channels, int rate)
  {
    if (outputLine != null || this.rate != rate || this.channels != channels)
    {
      if (outputLine != null)
      {
        outputLine.drain();
        outputLine.stop();
        outputLine.close();
      }
      init_audio(channels, rate);
      outputLine.start();
    }
    return outputLine;
  }

  // init audio

  private void init_audio(int channels, int rate)
  {
    try
    {
      AudioFormat audioFormat = new AudioFormat((float) rate, 16, channels, true, false);
      DataLine.Info info = new DataLine.Info(SourceDataLine.class, audioFormat, AudioSystem.NOT_SPECIFIED);
      if (!AudioSystem.isLineSupported(info))
        return;

      try
      {
        outputLine = (SourceDataLine) AudioSystem.getLine(info);
        outputLine.open(audioFormat);
      } catch (LineUnavailableException ex) {
        return;
      } catch (IllegalArgumentException ex) {
        return;
      }

      frameSizeInBytes = audioFormat.getFrameSize();
      int bufferLengthInFrames = outputLine.getBufferSize() / frameSizeInBytes / 2;
      bufferLengthInBytes = bufferLengthInFrames * frameSizeInBytes;

      this.rate = rate;
      this.channels = channels;
    } catch (Exception ee) { }
  }

  /**
   * Plays the OGG file.
   */

  public void run()
  {
    init_jorbis();
    Thread me = Thread.currentThread();

loop:
    while (true)
    {
      int eos = 0;

      int index = oy.buffer(BUFSIZE);
      buffer = oy.data;
      try
      {
        bytes = bitStream.read(buffer, index, BUFSIZE);
      } catch (Exception e) {
        return;
      }
      oy.wrote(bytes);

      if (oy.pageout(og) != 1)
      {
        if (bytes < BUFSIZE)
          break;
        //System.err.println("Input does not appear to be an Ogg bitstream.");
        fireError("Input does not appear to be an Ogg bitstream.");
        return;
      }

      os.init(og.serialno());
      os.reset();

      vi.init();
      vc.init();

      if (os.pagein(og) < 0)
      {
        //System.err.println("Error reading first page of Ogg bitstream data.");
        fireError("Error reading first page of Ogg bitstream data.");
        return;
      }

      if (os.packetout(op) != 1)
      {
        //System.err.println("Error reading initial header packet.");
        fireError("Error reading initial header packet.");
        break;
      }

      if (vi.synthesis_headerin(vc, op) < 0)
      {
        //System.err.println("This Ogg bitstream does not contain Vorbis audio data.");
        fireError("This Ogg bitstream does not contain Vorbis audio data.");
        return;
      }

      int i = 0;

      while (i < 2)
      {
        while (i < 2)
        {
          int result = oy.pageout(og);
          if (result == 0)
            break; // Need more data
          if (result == 1)
          {
            os.pagein(og);
            while (i < 2)
            {
              result = os.packetout(op);
              if (result == 0)
                break;
              if (result == -1)
              {
                //System.err.println("Corrupt secondary header. Exiting.");
                //return;
                fireError("Corrupt secondary header. Exiting.");
                break loop;
              }
              vi.synthesis_headerin(vc, op);
              i++;
            }
          }
        }

        index = oy.buffer(BUFSIZE);
        buffer = oy.data;
        try
        {
          bytes = bitStream.read(buffer, index, BUFSIZE);
        } catch (Exception e) {
          return;
        }

        if (bytes == 0 && i < 2)
        {
          //System.err.println("End of file before finding all Vorbis headers!");
          fireError("End of file before finding all Vorbis headers!");
          return;
        }
        oy.wrote(bytes);
      }

      // comments
      byte[][] cptr = vc.user_comments;
      for (int j = 0; j < cptr.length; j++)
      {
        if (cptr[j] == null)
          break;

        String comment = new String(cptr[j], 0, cptr[j].length - 1);
        if (comment.toLowerCase().startsWith("artist="))
        {
          if (listener != null)
            listener.setArtist(comment.substring(comment.indexOf('=') + 1));
        } else if (comment.toLowerCase().startsWith("album=")) {
          if (listener != null)
            listener.setAlbum(comment.substring(comment.indexOf('=') + 1));
        } else if (comment.toLowerCase().startsWith("title=")) {
          if (listener != null)
            listener.setTitle(comment.substring(comment.indexOf('=') + 1));
        }
      }
      cptr = null;

      //System.err.println("Bitstream is " + vi.channels + " channel, " + vi.rate + "Hz");
      //System.err.println("Encoded by: " + new String(vc.vendor, 0, vc.vendor.length - 1) + "\n");

      convsize = BUFSIZE / vi.channels;

      vd.synthesis_init(vi);
      vb.init(vd);

      double[][][]_pcm = new double[1][][];
      float[][][]_pcmf = new float[1][][];
      int[]_index = new int[vi.channels];

      getOutputLine(vi.channels, vi.rate);

      while (eos == 0)
      {
        while (eos == 0)
        {
          if (player != me)
          {
            try
            {
              bitStream.close();
            } catch (Exception ee) { }
            return;
          }

          int result = oy.pageout(og);
          if (result == 0)
            break; // need more data

          os.pagein(og);
          while (true)
          {
            result = os.packetout(op);
            if (result == 0)
              break; // need more data

            int samples;
            if (vb.synthesis(op) == 0)
            { // test for success!
              vd.synthesis_blockin(vb);
            }

            while ((samples = vd.synthesis_pcmout(_pcmf, _index)) > 0)
            {
              double[][] pcm = _pcm[0];
              float[][] pcmf = _pcmf[0];
              //boolean clipflag = false;
              int bout = (samples < convsize ? samples : convsize);

              // convert doubles to 16 bit signed ints (host order) and
              // interleave
              for (i = 0; i < vi.channels; i++)
              {
                int ptr = i * 2;
                //int ptr=i;
                int mono = _index[i];
                for (int j = 0; j < bout; j++)
                {
                  int val = (int) ((pcmf[i][mono + j] * 32767.) * volumeMultiplier);
                  if (val > 32767)
                  {
                    val = 32767;
                    //clipflag = true;
                  }

                  if (val < -32768)
                  {
                    val = -32768;
                    //clipflag = true;
                  }

                  if (val < 0)
                    val = val | 0x8000;
                  convbuffer[ptr] = (byte) (val);
                  convbuffer[ptr + 1] = (byte) (val >>> 8);
                  ptr += 2 * (vi.channels);
                }
              }

              outputLine.write(convbuffer, 0, 2 * vi.channels * bout);
              // TIMER IT !!
              second = (int) outputLine.getMicrosecondPosition() / 1000000;
              vd.synthesis_read(bout);
            }

            while (paused)
            {
              try
              {
                Thread.sleep(200);
              } catch (InterruptedException iex) { }
            }
          }
          if (og.eos() != 0)
            eos = 1;
        }

        if (eos == 0)
        {
          index = oy.buffer(BUFSIZE);
          buffer = oy.data;
          try
          {
            bytes = bitStream.read(buffer, index, BUFSIZE);
          } catch (Exception e) {
            System.err.println(e);
            return;
          }

          if (bytes == -1)
            break;
          oy.wrote(bytes);
          if (bytes == 0)
            eos = 1;
        }
      }

      os.clear();
      vb.clear();
      vd.clear();
      vi.clear();
    }

    oy.clear();

    try
    {
      if (bitStream != null)
        bitStream.close();
      player = null;
    } catch (Exception e) { }

    next();
  }

  /**
   * Stops the playing thread.
   */

  public void stop()
  {
    if (player == null)
    {
      try
      {
        //outputLine.drain();
        outputLine.flush();
        outputLine.stop();
        outputLine.close();
        if (bitStream != null)
          bitStream.close();
      } catch (Exception e) { }
    } else
      player = null;

    timer = null;
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // TIMER AND OGGLISTENER
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  // fires error messages to listener

  private void fireError(String errorMessage)
  {
    if (listener != null)
      listener.error(errorMessage);
  }

  /**
   * Sets the timer listener which will receive time elapsed informations.
   * @param listener The listener to be used
   */

  public void setOGGListener(OGGListener listener)
  {
    this.listener = listener;
  }

  // thread which generates the time elapsed event
 
  class Timer extends Thread
  {
    Timer()
    {
      super("OGGre timer");
      listener.timeElapsed(0);
      start();
    }

    public void run()
    {
      while(true)
      {
        if (second > lastSecond)
        {
          lastSecond = second;
          listener.timeElapsed(lastSecond);
        }

        try
        {
          Thread.sleep(600);
        } catch (InterruptedException iex) { }
      }
    }
  }

  // allows to quickly test the play of a play list

  public static void main(String[] args)
  {
    PlayList playList = PlayListFactory.getPlayListFromFile(System.getProperty("user.dir")  + "\\" + args[0]);
    Player player = new Player(playList);
    player.play();
  }
}

// End of Player.java

