/*
 * 07/08/2001 - 14:41:59
 *
 * PlayList.java - Playlist for the OGGre player
 * Copyright (C) 2001 Romain Guy
 * romain.guy@jext.org
 * www.jext.org
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package org.jext.oggre.playlist;

import java.util.ArrayList;
import java.util.Random;

/**
 * The <code>PlayList</code> objet holds informations about song meant to
 * be played by the OGG player.
 * @author Romain Guy
 */

public class PlayList
{
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PRIVATE FIELDS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  // contains the playlist items
  private ArrayList songs;
  // shuffle databases
  private ArrayList shuffleDatabase;
  private ArrayList shuffleDatabaseTrash;
  // trash index
  private int backTrash;
  // randomize engine
  private Random shuffler;
  // randomize play
  private boolean shuffle = false;
  // repeated play
  private boolean repeat = false;
  // current song
  private int currentSong = -1;

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // CONSTRUCTORS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Creates a new empty playlist.
   */

  public PlayList()
  {
    this(null);
  }

  /**
   * Creates a new playlist containing a given array of
   * songs. A song can be local (file) or distant (URL).
   * @param songsList An array of <code>Song</code> instances
   */

  public PlayList(Song[] songsList)
  {
    if (songsList != null)
      setSongs(songsList);
    else
      songs = new ArrayList();
  }

  // returns true if the song index is valid

  private boolean isValidSongIndex(int index)
  {
    return index >= 0 && index < songs.size();
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PLAY METHDODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns the next song in the playlist.
   */

  public Song next()
  {
    if (!shuffle)
    {
      currentSong++;
      if (!isValidSongIndex(currentSong))
      {
        currentSong = 0;

        if (repeat)
          return (Song) songs.get(0);
        else
          return null;
      }
    } else {
      if (backTrash == shuffleDatabaseTrash.size())
      {
        int size = shuffleDatabase.size();
        if (size == 0)
        {
          if (repeat)
          {
            buildShuffle();
            size = shuffleDatabase.size();
          } else
            return null;
        }

        int song = shuffler.nextInt(size);
        currentSong = ((Integer) shuffleDatabase.get(song)).intValue();

        shuffleDatabase.remove(song);
        shuffleDatabaseTrash.add(new Integer(currentSong));
        backTrash++;
      } else {
        currentSong = ((Integer) shuffleDatabaseTrash.get(backTrash)).intValue();
        backTrash++;
      }
    }

    return (Song) songs.get(currentSong);
  }

  /**
   * Returns the previous song in the playlist.
   */

  public Song previous()
  {
    if (!shuffle)
    {
      currentSong--;
      if (!isValidSongIndex(currentSong))
      {
        currentSong = songs.size() - 1;

        if (repeat)
          return (Song) songs.get(currentSong);
        else
          return null;
      }
    } else {
      if (backTrash == 0)
      {
        if (repeat)
        {
          buildShuffle();
          return next();
        } else
          return null;
      } else {
        backTrash--;
        int song = ((Integer) shuffleDatabaseTrash.get(backTrash)).intValue();
        if (currentSong == song && backTrash > 0)
        {
          backTrash--;
          currentSong = ((Integer) shuffleDatabaseTrash.get(backTrash)).intValue();
        } else
          currentSong = song;
      }
    }

    return (Song) songs.get(currentSong);
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // SUFFLE METHDODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns true if the shuffle option is activated.
   */

  public boolean isShuffleEnabled()
  {
    return shuffle;
  }

  /**
   * Sets the shuffle option.
   * @param shuffle The value of the shuffle option
   */

  public void setShuffleEnabled(boolean shuffle)
  {
    this.shuffle = shuffle;
    if (shuffle)
      buildShuffle();
  }

  // builds the shuffle database

  private void buildShuffle()
  {
    int size = songs.size();
    shuffleDatabase = new ArrayList(size);
    shuffleDatabaseTrash = new ArrayList(size);
    backTrash = 0;

    for (int i = 0; i < size; i++)
      shuffleDatabase.add(new Integer(i));

    // lazy instanciation
    if (shuffler == null)
      shuffler = new Random();
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // REPEAT METHDODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns true if the repeat option is activated.
   */

  public boolean isRepeatEnabled()
  {
    return repeat;
  }

  /**
   * Sets the repeat option.
   * @param repeat The value of the repeat option
   */

  public void setRepeatEnabled(boolean repeat)
  {
    this.repeat = repeat;
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // PLAYLIST METHDODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Adds a song to the play list.
   * @param song The song to be added
   */

  public void addSong(Song song)
  {
    if (song != null)
    {
      songs.add(song);
      if (shuffle)
        buildShuffle();
    }
  }

  /**
   * Returns the current selected song.
   */

  public Song getCurrentSong()
  {
    return ((currentSong != -1 && isValidSongIndex(currentSong)) ? (Song) songs.get(currentSong) : null);
  }

  /**
   * Returns a given song.
   */

  public Song getSongAt(int index)
  {
    return (isValidSongIndex(index) ? (Song) songs.get(index) : null);
  }

  /**
   * Removes a song from the playlist.
   * @param song The song to be removed
   */

  public void removeSong(Song song)
  {
    if (song != null)
    {
      songs.remove(song);
      if (shuffle)
        buildShuffle();
    }
  }

  /**
   * Removes a song from the playlist given its name.
   * @param song The name of the song to be removed
   */

  public void removeSong(String song)
  {
    if (song == null)
      return;
    Song _song = null;

    for (int i = 0; i < songs.size(); i++)
    {
      _song = (Song) songs.get(i);
      if (song.equals(_song.getName()))
      {
        songs.remove(i);
        break;
      }
    }

    if (shuffle)
      buildShuffle();
  }

  /**
   * Sets the current selected song.
   * @param currentSong The index of the current song
   */

  public void setCurrentSong(int currentSong)
  {
    if (isValidSongIndex(currentSong))
      this.currentSong = currentSong;
  }

  ////////////////////////////////////////////////////////////////////////////////////////////////////////////
  // ACCESSIBILITY METHDODS
  ////////////////////////////////////////////////////////////////////////////////////////////////////////////

  /**
   * Returns an <code>ArrayList</code> holding all the playlist songs.
   */

  public ArrayList getPlayListArrayList()
  {
    return songs;
  }

  /**
   * Returns playlist contents as an array of songs.
   */

  public Song[] getSongs()
  {
    Object[] o = songs.toArray();
    Song[] songsArray = new Song[o.length];

    for (int i = 0; i < o.length; i++)
      songsArray[i] = (Song) o[i];

    return songsArray;
  }

  /**
   * Sets the playlist contents.
   * @param songsList The new array of songs
   */

  public void setSongs(Song[] songsList)
  {
    songs = new ArrayList(songsList.length);
    for (int i = 0; i < songsList.length; i++)
      songs.add(songsList[i]);

    currentSong = 0;
    if (shuffle)
      buildShuffle();
  }
}

// End of PlayList.java

